/**
 * \file: mspin_connection_tls_server.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * mySPIN Connection Adapter TCP/IP
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel BSOT/PJ-ES1 thilo.fickel@bosch-softtec.com
 *
 * @copyright: (c) 2016 Bosch SoftTec GmbH
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_connection_tls_server.h"
#include "mspin_connection_adapter.h"
#include "mspin_connection_tcp_server.h"
#include "mspin_connection_tcp_manager.h"
#include "mspin_connection_tcp_listener.h"
#include "mspin_logging.h"

#include <string.h>         //memset, strerror
#include <pthread.h>        //pthread_*
#include <errno.h>          //errno
#include <sys/prctl.h>      //prctl
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <openssl/err.h>    //ERR_free_strings

#define MSPIN_TLS_ENABLE_MULTITHREADING_SUPPORT //Enable for OpenSSL multithreading support

#ifdef MSPIN_TLS_ENABLE_MULTITHREADING_SUPPORT

typedef struct CRYPTO_dynlock_value CRYPTO_dynlock_value_t;

struct CRYPTO_dynlock_value {
    pthread_mutex_t mutex;
};

typedef struct
{
    pthread_mutex_t *pSSLMutexArray;
    int numberLocks;
} mspin_tls_threadLocks_t;

static mspin_tls_threadLocks_t gThreadLocks = { NULL, 0 };

#endif //MSPIN_TLS_ENABLE_MULTITHREADING_SUPPORT

static BOOL gSSLLibraryInitialized = FALSE;

#ifdef MSPIN_TLS_ENABLE_MULTITHREADING_SUPPORT
static void mspin_tls_sslThreadId(CRYPTO_THREADID *id)
{
//    mspin_log_printLn(eMspinVerbosityVerbose, "%s() called for %lu", __FUNCTION__, (unsigned long int)pthread_self());

    CRYPTO_THREADID_set_pointer(id, (void*) pthread_self());
}

static void mspin_tls_deinitMutexArray(void)
{
    int i = 0;

    if (gThreadLocks.pSSLMutexArray)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() Freeing SSLMutex=%p with %d locks",
                __FUNCTION__, gThreadLocks.pSSLMutexArray, gThreadLocks.numberLocks);

        for (i = 0; i < gThreadLocks.numberLocks; i++)
        {
            pthread_mutex_destroy(&(gThreadLocks.pSSLMutexArray[i]));
        }

        free(gThreadLocks.pSSLMutexArray);
        gThreadLocks.pSSLMutexArray = NULL;
        gThreadLocks.numberLocks = 0;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s() WARNING: No SLL mutex array ", __FUNCTION__);
    }
}

static int mspin_tls_initMutexArray(void)
{
    int n = 0;
    int ret = 0;

    if (gThreadLocks.pSSLMutexArray == NULL)
    {
        gThreadLocks.numberLocks = CRYPTO_num_locks();

        gThreadLocks.pSSLMutexArray = malloc(gThreadLocks.numberLocks*sizeof(pthread_mutex_t));
        if (gThreadLocks.pSSLMutexArray)
        {
            for (n = 0; n < gThreadLocks.numberLocks; n++)
            {
                if (pthread_mutex_init(&(gThreadLocks.pSSLMutexArray[n]), NULL))
                {
                    ret = -2;
                }
            }
            mspin_log_printLn(eMspinVerbosityDebug, "%s() %d mutex allocated, return %d", __FUNCTION__,
                    gThreadLocks.pSSLMutexArray, ret);

            return ret;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to allocate memory", __FUNCTION__);
            return -1;
        }
    }
    else
    {

        mspin_log_printLn(eMspinVerbosityWarn, "%s() WARNING: Already initialized", __FUNCTION__);

        return 1;
    }
}

/*PRQA: Lint Message 454, 455, 456: CRYPTO_LOCK is checked before lock or unclock the mutex. So ignore these errors.*/
/*lint -e454 -e455 -e456*/
static void mspin_tls_sslLock(int mode, int mutexIndex, const char *file, int line)
{
    mspin_log_printLn(eMspinVerbosityVerbose, "%s(mode=%s, index=%d, location='%s:%d') called ",
            __FUNCTION__, (mode & CRYPTO_LOCK) ? "lock" : "unlock", mutexIndex, file, line);
    if (NULL != gThreadLocks.pSSLMutexArray)
    {
        if(mode & CRYPTO_LOCK)
        {
            pthread_mutex_lock(&(gThreadLocks.pSSLMutexArray[mutexIndex]));
        }
        else
        {
            pthread_mutex_unlock(&(gThreadLocks.pSSLMutexArray[mutexIndex]));
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(mode=%s, index=%d, location='%s:%d') ERROR: No SSL mutex ",
                __FUNCTION__, (mode & CRYPTO_LOCK) ? "lock" : "unlock", mutexIndex, file, line);
    }
}
/*lint +e454 +e455 +e456*/

static CRYPTO_dynlock_value_t *mspin_tls_sslDynlockCreate(const char *file, int line)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(location='%s:%d') called ", __FUNCTION__, file, line);

    CRYPTO_dynlock_value_t *pLock = malloc(sizeof(CRYPTO_dynlock_value_t));
    if (pLock)
    {
        if (pthread_mutex_init(&(pLock->mutex), NULL))
        {
            // Initialization of mutex failed
            mspin_log_printLn(eMspinVerbosityError, "%s(location='%s:%d') ERROR: Initialize of mutex failed", __FUNCTION__, file, line);

            free(pLock);
            return NULL;
        }
        else
        {
            return pLock;
        }
    }
    else
    {
        return NULL;
    }
}

static void mspin_tls_sslDynlockDestroy(CRYPTO_dynlock_value_t *pLock, const char *file, int line)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(pLock=%p, location='%s:%d') called ", __FUNCTION__, pLock, file, line);

    if (pLock)
    {
        pthread_mutex_destroy(&pLock->mutex);
        free(pLock);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(pLock=%p, location='%s:%d') FATAL ERROR: pLock is NULL ",
                __FUNCTION__, pLock, file, line);
    }
}
/*PRQA: Lint Message 454, 455, 456: CRYPTO_LOCK is checked before lock or unclock the mutex. So ignore these errors.*/
/*lint -e454 -e455 -e456*/
static void mspin_tls_sslDynlockLock(int mode, CRYPTO_dynlock_value_t *pLock, const char *file, int line)
{
    MSPIN_UNUSED(file);
    MSPIN_UNUSED(line);
//    mspin_log_printLn(eMspinVerbosityVerbose, "%s(mode=%s, location='%s:%d') called ", __FUNCTION__,
//            (mode & CRYPTO_LOCK) ? "lock" : "unlock", file, line);

    if (mode & CRYPTO_LOCK)
    {
        pthread_mutex_lock(&(pLock->mutex));
    }
    else
    {
        pthread_mutex_unlock(&pLock->mutex);
    }
}
/*lint +e454 +e455 +e456*/
#endif //MSPIN_TLS_ENABLE_MULTITHREADING_SUPPORT

void mspin_tls_initTLS(void)
{
#if OPENSSL_VERSION_NUMBER < 0x10100000L
    //Call is only required once. When freeing strings with 'mspin_tls_deinitTLS', this has to be repeated.
    //This call is only required when using OpenSSL before version 1.1
    SSL_load_error_strings();
#endif //OPENSSL_VERSION_NUMBER < 0x10100000L

    if (!gSSLLibraryInitialized)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() init SSL library", __FUNCTION__);
        gSSLLibraryInitialized = TRUE;
        SSL_library_init();
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() SSL library already initialized. Only load error strings",
                __FUNCTION__);
    }

#ifdef MSPIN_TLS_ENABLE_MULTITHREADING_SUPPORT
    if (mspin_tls_initMutexArray() < 0)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: mspin_tls_initMutexArray() failed -> do not set thread locks", __FUNCTION__);
        return;
    }

    CRYPTO_THREADID_set_callback(mspin_tls_sslThreadId);
    CRYPTO_set_locking_callback(mspin_tls_sslLock);
    CRYPTO_set_dynlock_create_callback(&mspin_tls_sslDynlockCreate);
    CRYPTO_set_dynlock_destroy_callback(&mspin_tls_sslDynlockDestroy);
    CRYPTO_set_dynlock_lock_callback(&mspin_tls_sslDynlockLock);
#endif //MSPIN_TLS_ENABLE_MULTITHREADING_SUPPORT
}

//ToDo: Right now this will never be called => so gpSSLMutexArray will never be released
void mspin_tls_deinitTLS(void)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s() entered", __FUNCTION__);

#ifdef MSPIN_TLS_ENABLE_MULTITHREADING_SUPPORT
    mspin_tls_deinitMutexArray();
#endif //MSPIN_TLS_ENABLE_MULTITHREADING_SUPPORT

#if OPENSSL_VERSION_NUMBER < 0x10100000L
    mspin_log_printLn(eMspinVerbosityDebug, "%s() free error strings", __FUNCTION__);
    //This call is only required when using OpenSSL before version 1.1
    ERR_free_strings(); //release strings loaded with SSL_load_error_strings
#endif //OPENSSL_VERSION_NUMBER < 0x10100000L
}

mspin_tls_parameter_t* mspin_tls_createContext(void)
{
    mspin_tls_parameter_t* pTLSContext = NULL;

    pTLSContext = malloc(sizeof(mspin_tls_parameter_t));

    if (!pTLSContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to allocate memory for TLS context",
                __FUNCTION__);
        return NULL;
    }

    memset(pTLSContext, 0, sizeof(mspin_tls_parameter_t));

    mspin_log_printLn(eMspinVerbosityDebug, "%s() TLS context=%p created", __FUNCTION__, pTLSContext);

    return pTLSContext;
}

void mspin_tls_deleteContext(mspin_tls_parameter_t** ppTLSContext)
{
    mspin_tls_parameter_t* pTLSContext = *ppTLSContext;

    if (pTLSContext)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(listener=%p) deleting SSL_CTX...", __FUNCTION__, pTLSContext);

        (void)mspin_tls_deleteSSLContext(pTLSContext); //we do not care about the result. Another debug print is already present

        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(listener=%p) deleting strings (caFile=%p, certFile=%p, keyFile=%p) ...",
                __FUNCTION__, pTLSContext, pTLSContext->caFile, pTLSContext->certFile, pTLSContext->keyFile);

        //Release memory of all allocated strings for certificate, private key and CA
        if (pTLSContext->caFile)
        {
            free(pTLSContext->caFile);
        }

        if (pTLSContext->certFile)
        {
            free(pTLSContext->certFile);
        }

        if (pTLSContext->keyFile)
        {
            free(pTLSContext->keyFile);
        }

        mspin_log_printLn(eMspinVerbosityDebug, "%s(listener=%p) deleting TLS context...", __FUNCTION__, pTLSContext);

        free(pTLSContext);
        *ppTLSContext = NULL;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s(listener=%p) WARNING: TLS context is NULL, nothing to do",
                __FUNCTION__, pTLSContext);
    }
}

MSPIN_ERROR mspin_tls_setParameter(mspin_tls_parameter_t* pTLSContext, MSPIN_TLS_CONFIGURATION_t* pTLSConfig)
{
    if (!pTLSConfig)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(pTLSConfig=%p) ERROR: pTLSConfig is NULL",
                __FUNCTION__, pTLSConfig);
        return MSPIN_ERROR_NULL_HANDLE;
    }

    if (!pTLSContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(pTLSConfig=%p) ERROR: pTLSContext is NULL",
                __FUNCTION__, pTLSConfig);
        return MSPIN_ERROR_NULL_HANDLE;
    }

    //Copy certificate filename
    pTLSContext->certFile = (char*)malloc((strlen(pTLSConfig->certificate)+1)*sizeof(char));
    mspin_log_printLn(eMspinVerbosityDebug, "%s(pTLSConfig=%p) buffer for certFile is %d",
            __FUNCTION__, pTLSConfig, (strlen(pTLSConfig->certificate)+1)*sizeof(char));

    strcpy(pTLSContext->certFile, pTLSConfig->certificate);

    //Copy private key filename
    pTLSContext->keyFile = (char*)malloc((strlen(pTLSConfig->privateKey)+1)*sizeof(char));
    strcpy(pTLSContext->keyFile, pTLSConfig->privateKey);

    mspin_log_printLn(eMspinVerbosityDebug, "%s(pTLSConfig=%p) using keyfile='%s' (len=%d, ptr=%p)",
            __FUNCTION__, pTLSConfig, pTLSContext->keyFile ? pTLSContext->keyFile : "n/a",
            pTLSContext->keyFile ? strlen(pTLSContext->keyFile) : 0,
            pTLSContext->keyFile ? pTLSContext->keyFile : 0);

    //Copy CA filename
    pTLSContext->caFile = (char*)malloc((strlen(pTLSConfig->ca)+1)*sizeof(char));
    strcpy(pTLSContext->caFile, pTLSConfig->ca);

    mspin_log_printLn(eMspinVerbosityDebug, "%s(pTLSConfig=%p) using CA file='%s' (len=%d, ptr=%p)",
            __FUNCTION__, pTLSConfig, pTLSContext->caFile ? pTLSContext->caFile : "n/a",
            pTLSContext->caFile ? strlen(pTLSContext->caFile) : 0,
            pTLSContext->caFile ? pTLSContext->caFile : 0);

    //Set options
    pTLSContext->limitToAES256 = pTLSConfig->aes256only;
    pTLSContext->verifyClient = pTLSConfig->verifyClient;

    return MSPIN_SUCCESS;
}

MSPIN_ERROR mspin_tls_createSSLContext(mspin_tls_parameter_t* pTLSContext)
{
    const SSL_METHOD *method = NULL;

    if (!pTLSContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(TLSContext=%p) ERROR: pTLSContext is NULL",
                __FUNCTION__, pTLSContext);
        return MSPIN_ERROR_NULL_HANDLE;
    }

    //Check if SSLContext is still present. Should not happen!
    if (pTLSContext->pSSLContext)
    {
#if OPENSSL_VERSION_NUMBER >= 0x10100000L
        mspin_log_printLn(eMspinVerbosityWarn, "%s(TLSContext=%p) WARNING: tlsContext is still present (not NULL) -> call SSL_CTX_free to decrement ref count",
                __FUNCTION__, pTLSContext);

        SSL_CTX_free(pTLSContext->pSSLContext); //decrements reference count
        pTLSContext->pSSLContext = NULL;
#else
        mspin_log_printLn(eMspinVerbosityWarn, "%s(TLSContext=%p) WARNING: tlsContext is still present (not NULL) -> free",
                __FUNCTION__, pTLSContext);
        SSL_CTX_free(pTLSContext->pSSLContext); //frees the SSL context
        pTLSContext->pSSLContext = NULL;
#endif //OPENSSL_VERSION_NUMBER >= 0x10100000L
    }
    //Get TLS v1.2 method
#if OPENSSL_VERSION_NUMBER >= 0x10100000L
    mspin_log_printLn(eMspinVerbosityDebug, "%s(TLSContext=%p) Using TLS_server_method() (OpenSSL 1.1 or higher)",
            __FUNCTION__, pTLSContext);
    method = TLS_server_method();
#else

    mspin_log_printLn(eMspinVerbosityWarn, "%s(TLSContext=%p) WARNING: Using TLSv1_2_server_method() (OpenSSL < 1.1)",
            __FUNCTION__, pTLSContext);
    method = TLSv1_2_server_method();
#endif //OPENSSL_VERSION_NUMBER >= 0x10100000L

    if (!method)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(TLSContext=%p) ERROR: Setting server method failed NULL",
                __FUNCTION__, pTLSContext);
        return MSPIN_ERROR_GENERAL;
    }

    //Create new SSL_CTX
    pTLSContext->pSSLContext = SSL_CTX_new(method);
    if (!pTLSContext->pSSLContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(TLSContext=%p) ERROR: Failed to create new SSL context",
                __FUNCTION__, pTLSContext);
        return MSPIN_ERROR_GENERAL;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(TLSContext=%p) context=%p created",
            __FUNCTION__, pTLSContext, pTLSContext->pSSLContext);

    return MSPIN_SUCCESS;
}

MSPIN_ERROR mspin_tls_deleteSSLContext(mspin_tls_parameter_t* pTLSContext)
{
    //Check TLS Context
    if (!pTLSContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(TLSContext=%p) ERROR: pTLSContext is NULL",
                __FUNCTION__, pTLSContext);
        return MSPIN_ERROR_NULL_HANDLE;
    }

    if (pTLSContext->pSSLContext)
    {
#if OPENSSL_VERSION_NUMBER >= 0x10100000L
        mspin_log_printLn(eMspinVerbosityDebug, "%s(TLSContext=%p) call SSL_CTX_free to decrement ref count of %p",
                __FUNCTION__, pTLSContext, pTLSContext->pSSLContext);

        SSL_CTX_free(pTLSContext->pSSLContext); //decrements reference count
        pTLSContext->pSSLContext = NULL;
#else
        mspin_log_printLn(eMspinVerbosityDebug, "%s(TLSContext=%p) free %p",
                __FUNCTION__, pTLSContext, pTLSContext->pSSLContext);
        SSL_CTX_free(pTLSContext->pSSLContext); //frees the SSL context
        pTLSContext->pSSLContext = NULL;
#endif //OPENSSL_VERSION_NUMBER >= 0x10100000L
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(TLSContext=%p) SSL_CTX is NULL", __FUNCTION__, pTLSContext);
    }

    return MSPIN_SUCCESS;
}

MSPIN_ERROR mspin_tls_configureSSLContext(mspin_tls_parameter_t* pTLSContext)
{
    if (!pTLSContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(TLSContext=%p) ERROR: pTLSContext is NULL",
                __FUNCTION__, pTLSContext);
        return MSPIN_ERROR_NULL_HANDLE;
    }

    //SSL_CTX must be also present -> check
    if (!pTLSContext->pSSLContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(TLSContext=%p) FATAL ERROR: pSSLContext is NULL",
                __FUNCTION__, pTLSContext);
        return MSPIN_ERROR_NOT_READY;
    }

    //Limit cipher list to AES256-SHA?
    if (pTLSContext->limitToAES256)
    {
        if (0 == SSL_CTX_set_cipher_list(pTLSContext->pSSLContext, "AES256-SHA"))
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(TLSContext=%p) ERROR: Failed to set cipher with error='%s'",
                    __FUNCTION__, pTLSContext, ERR_error_string(ERR_get_error(), NULL));
            return MSPIN_ERROR_GENERAL;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(TLSContext=%p) Cipher list limited to AES256-SHA only",
                    __FUNCTION__, pTLSContext);
        }
    }

    //Do not use automatic curve selection
    //SSL_CTX_set_ecdh_auto(pCTX, 1); //this sets automatic curve selection for server CTX or SSL to on/off,
    // see https://www.openssl.org/docs/man1.0.2/ssl/SSL_set_ecdh_auto.html

    //Set the key and certificate
    if (SSL_CTX_use_certificate_file(pTLSContext->pSSLContext, pTLSContext->certFile, SSL_FILETYPE_PEM) < 0)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(TLSContext=%p) ERROR: Setting key failed with '%s'",
                __FUNCTION__, pTLSContext, ERR_error_string(ERR_get_error(), NULL));
        return MSPIN_ERROR_GENERAL;
    }

    if (SSL_CTX_use_PrivateKey_file(pTLSContext->pSSLContext, pTLSContext->keyFile, SSL_FILETYPE_PEM) < 0 )
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(TLSContext=%p) ERROR: Loading private key failed with '%s'",
                __FUNCTION__, pTLSContext, ERR_error_string(ERR_get_error(), NULL));
        return MSPIN_ERROR_GENERAL;
    }

    //Authenticate client by requesting client certificate
    if (pTLSContext->verifyClient)
    {
        //Load CA root file for verifying client
        if (strlen(pTLSContext->caFile) > 0)
        {
            if (1 != SSL_CTX_load_verify_locations(pTLSContext->pSSLContext, pTLSContext->caFile, NULL))
            {
                mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to load verify locations '%s' with '%s'",
                        __FUNCTION__, pTLSContext->caFile, ERR_error_string(ERR_get_error(), NULL));
                return MSPIN_ERROR_GENERAL;
            }
            mspin_log_printLn(eMspinVerbosityInfo, "%s(TLSContext=%p) Root CA '%s' loaded",
                    __FUNCTION__, pTLSContext, pTLSContext->caFile);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(TLSContext=%p) ERROR: No Root CA specified (empty). Client authentication will fail!",
                    __FUNCTION__, pTLSContext);
        }

        //And request client verification
        SSL_CTX_set_verify(pTLSContext->pSSLContext, SSL_VERIFY_PEER | SSL_VERIFY_FAIL_IF_NO_PEER_CERT, NULL); //ToDo: Add SSL_VERIFY_CLIENT_ONCE flag to query client certificate only once?
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(TLSContext=%p) done", __FUNCTION__, pTLSContext);
    return MSPIN_SUCCESS;
}

MSPIN_ERROR mspin_tls_connectClient(mspin_context_t* pContext, S32 fd)
{
    in_addr_t ipAddr = 0;
    mspin_connectionHandle_t *pConnectionHandle = NULL;

    mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, connectionID=%d) entered", __FUNCTION__, pContext, fd);

    if (pContext)
    {
        //Release any existing connection handles
        mspin_conn_releaseHandle(&(pContext->pConnectionHandle));

        //Create connection handle
        pConnectionHandle = mspin_conn_createHandle();
        if (!pConnectionHandle)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, id=%d) ERROR: Failed to create connection handle",
                 __FUNCTION__, pContext, fd);

            //Get IP address from connection
            ipAddr = mspin_tcp_getIPAddr(fd);

            (void)mspin_tcp_closeConnection(fd);
            mspin_tcp_signalConnectionClosed(fd, ipAddr, MSPIN_TCP_CONNECTION_OTHER_ERROR);

            return MSPIN_ERROR_MEM_ALLOC;
        }

        pConnectionHandle->connectionID = fd;
        pConnectionHandle->connectionType = MSPIN_CONNECTION_SERVER_TLS;
        pConnectionHandle->result = 0; //success
        pContext->pConnectionHandle = pConnectionHandle; //and assign it to the mySPIN context

        //Update status of connection
        if (!mspin_tcp_setActive(fd))
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, id=%d) ERROR: Could not find connection ID",
                 __FUNCTION__, pContext, fd);

            return MSPIN_ERROR_NOT_FOUND;
        }

        mspin_log_printLn(eMspinVerbosityDebug, "%s(ctx=%p, id=%d) done", __FUNCTION__, pContext, fd);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, socketFD=%d) ERROR: Connection handle is NULL",
                __FUNCTION__, pContext, fd);

        return MSPIN_ERROR_NULL_HANDLE;
    }

    return MSPIN_SUCCESS;
}

S32 mspin_tls_receive(mspin_connectionHandle_t *pConnectionHandle, U8* buffer, U32 bufferLen, U32 timeout)
{
    S32 rc = -1;
    SSL* pSSL = NULL;
    fd_set fds = {{0}};
    struct timeval tv;
    int selectResult = 0;
    int tcpSocket = -1;
    bool setError = FALSE;

    if (pConnectionHandle)
    {
        tcpSocket = pConnectionHandle->connectionID; //store in local variable which is valid till this iteration
        /* PRQA: Lint Message 529: Warning due symbol __do not subsequently referenced in the asm section. Not an issue */
        /*lint -save -e529*/
        FD_ZERO(&fds);
        /*lint -restore*/
        FD_SET(tcpSocket, &fds);
        tv.tv_sec = timeout/1000; //seconds
        tv.tv_usec = (timeout%1000)*1000; //milliseconds

        mspin_log_printLn(eMspinVerbosityVerbose, "%s(connHandle=%p, buffer=%p, len=%d) wait for data",
                __FUNCTION__, pConnectionHandle, buffer, bufferLen);

        selectResult = select(tcpSocket+1, &fds, NULL, NULL, &tv);
        if ((selectResult > 0) && FD_ISSET(tcpSocket, &fds))
        {
            //Read data
            mspin_log_printLn(eMspinVerbosityVerbose, "%s(connHandle=%p, buffer=%p, len=%d) call read",
                    __FUNCTION__, pConnectionHandle, buffer, bufferLen);

            pSSL = mspin_tcp_lockSSL(tcpSocket); //acquires connection lock, must be released with 'mspin_tcp_unlockSSL'
            if (pSSL)
            {
                rc = SSL_read(pSSL, (char*)buffer, bufferLen);

                if (rc > 0)
                {
                    mspin_log_printLn(eMspinVerbosityVerbose, "%s(connHandle=%p, buffer=%p, len=%d) %d bytes read",
                            __FUNCTION__, pConnectionHandle, buffer, bufferLen, rc);
                }
                else if (0 == rc)
                {
                    int error = SSL_get_error((const SSL*)pSSL, rc);
                    unsigned long detailedError = ERR_get_error();
                    if (SSL_ERROR_ZERO_RETURN == error)
                    {
                        mspin_log_printLn(eMspinVerbosityError,
                                "%s(connHandle=%p, buffer=%p, len=%d) ERROR: Connection shut down",
                                __FUNCTION__, pConnectionHandle, buffer, bufferLen);
                    }
                    else
                    {
                        mspin_log_printLn(eMspinVerbosityError,
                                "%s(connHandle=%p, buffer=%p, len=%d) ERROR: SSL_read failed with rc=%d and error=%d",
                                __FUNCTION__, pConnectionHandle, buffer, bufferLen, rc, error);
                        mspin_log_printLn(eMspinVerbosityError,
                                "%s(connHandle=%p, buffer=%p, len=%d) ERROR: Details are '%s'",
                                __FUNCTION__, pConnectionHandle, buffer, bufferLen,
                                ERR_error_string(detailedError, NULL));
                    }

                    //Set error to true
                    setError = TRUE;
                    rc = -1; //change return code to -1 in order to handle this as error
                }
                else
                {
                    int error = SSL_get_error((const SSL*)pSSL, rc);
                    unsigned long detailedError = ERR_get_error();

                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(connHandle=%p, buffer=%p, len=%d) ERROR: SSL_read failed with rc=%d and error=%d",
                            __FUNCTION__, pConnectionHandle, buffer, bufferLen, rc, error);
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(connHandle=%p, buffer=%p, len=%d) ERROR: Details are '%s'",
                            __FUNCTION__, pConnectionHandle, buffer, bufferLen,
                            ERR_error_string(detailedError, NULL));
                    setError = TRUE;
                    rc = -1; //return negative rc
                }

                mspin_tcp_unlockSSL(tcpSocket); //unlock connection again

                //Update error after unlock
                if (setError)
                {
                    mspin_tcp_setError(tcpSocket, TRUE);
                }
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, len=%d) ERROR: SSL* is NULL",
                        __FUNCTION__, pConnectionHandle, buffer, bufferLen);
                rc = -1; //return negative rc
            }
        }
        else if (selectResult < 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(connHandle=%p, buffer=%p, len=%d) ERROR: select() returned with error code=%d and errno='%s'(%d) => abort",
                    __FUNCTION__, pConnectionHandle, buffer, bufferLen, selectResult, strerror(errno), errno);
            rc = -1;
        }
        else if (selectResult == 0)
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(connHandle=%p, buffer=%p, len=%d) timeout",
                    __FUNCTION__, pConnectionHandle, buffer, bufferLen, rc);
            rc = 0;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, len=%d) ERROR: select() indicated data on invalid fd",
                    __FUNCTION__, pConnectionHandle, buffer, bufferLen, selectResult);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: TCP handle is NULL",
                __FUNCTION__, pConnectionHandle, buffer, bufferLen);
    }

    return rc;
}

S32 mspin_tls_send(mspin_connectionHandle_t *pConnectionHandle, const U8* buffer, U32 bufferLen)
{
    S32 rc = 0; //must be 0 to enter the loop
    U32 transferredUntilNow = 0;
    int tcpSocket = -1;
    SSL *pSSL = NULL;
    bool setError = FALSE;

    if (pConnectionHandle)
    {
        while ((bufferLen > transferredUntilNow) && (rc >= 0))
        {
            tcpSocket = pConnectionHandle->connectionID;
            if ((tcpSocket > -1) && !mspin_tcp_getError(tcpSocket))
            {
                pSSL = mspin_tcp_lockSSL(tcpSocket); //acquires connection lock, must be released with 'mspin_tcp_unlockSSL'
                if (pSSL)
                {
                    mspin_log_printLn(eMspinVerbosityVerbose, "%s(connHandle=%p, buffer=%p, bufferLen=%d) before sending",
                            __FUNCTION__, pConnectionHandle, buffer, bufferLen, transferredUntilNow);

                    rc = SSL_write(pSSL, buffer+transferredUntilNow, bufferLen-transferredUntilNow);

                    if (rc > 0)
                    {
                        transferredUntilNow += rc;
                        if (transferredUntilNow == bufferLen)
                        {
                            mspin_log_printLn(eMspinVerbosityDebug, "%s(connHandle=%p, buffer=%p, bufferLen=%d) all %d bytes sent",
                                    __FUNCTION__, pConnectionHandle, buffer, bufferLen, transferredUntilNow);
                        }
                        else if (transferredUntilNow < bufferLen)
                        {
                            mspin_log_printLn(eMspinVerbosityDebug, "%s(connHandle=%p, buffer=%p, bufferLen=%d) %d/%d bytes sent -> try sending remaining bytes",
                                    __FUNCTION__, pConnectionHandle, buffer, bufferLen, transferredUntilNow, bufferLen);
                        }
                        else
                        {
                            mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: Too many bytes=%d sent",
                                    __FUNCTION__, pConnectionHandle, buffer, bufferLen, rc);
                        }
                    }
                    else if (rc == 0)
                    {
                        //Write operation was not successful. Call SSL_get_error() to find out whether an error occurred or connection was shut down
                        int error = SSL_get_error((const SSL*)pSSL, rc);
                        if (SSL_ERROR_ZERO_RETURN == error)
                        {
                            mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: Connection shut down",
                                    __FUNCTION__, pConnectionHandle, buffer, bufferLen);
                        }
                        else
                        {
                            mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: Sending failed with rc=%d and error=%d",
                                    __FUNCTION__, pConnectionHandle, buffer, bufferLen, rc, error);
                        }

                        //Set error to true
                        setError = TRUE;
                    }
                    else
                    {
                        int error = SSL_get_error(pSSL, rc);
                        mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: Sending failed with rc=%d and error=%d",
                                __FUNCTION__, pConnectionHandle, buffer, bufferLen, rc, error);

                        //Set error to true
                        setError = TRUE;
                    }
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: SSL* is NULL",
                            __FUNCTION__, pConnectionHandle, buffer, bufferLen);
                    rc = -1;

                    //Set error to true
                    setError = TRUE;
                }

                //First unlock then set error if required
                mspin_tcp_unlockSSL(tcpSocket); //unlock connection again

                if (setError)
                {
                    mspin_tcp_setError(tcpSocket, TRUE); //set error flag
                }
            }
            else if (tcpSocket > -1)
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: Error set => do not sent",
                        __FUNCTION__, pConnectionHandle, buffer, bufferLen);
                rc = -1;
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: Invalid socket",
                        __FUNCTION__, pConnectionHandle, buffer, bufferLen);
                rc = -1;
            }
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(connHandle=%p, buffer=%p, bufferLen=%d) ERROR: TCP handle is NULL",
                __FUNCTION__, pConnectionHandle, buffer, bufferLen);
        rc = -1;
    }

    return rc;
}
